import os
import pygame

class HUD:
    def __init__(self, width, height, logdir):
        self.dim = (width, height)
        self.logdir = logdir + '/hud/'
        if not os.path.exists(self.logdir):
            os.makedirs(self.logdir, exist_ok=True)
        pygame.init()
        font_name = 'ubuntumono'
        font = pygame.font.match_font(font_name)
        self.font = pygame.font.Font(font, 14)
        self.display = pygame.display.set_mode(self.dim, pygame.HWSURFACE | pygame.DOUBLEBUF)
        pygame.display.set_caption('Central HUD')

        self._info_text = []
        self._server_clock = pygame.time.Clock()
        self._current_frame = None
    
    def on_world_tick(self, timestamp):
        self._server_clock.tick()
        self._server_fps = self._server_clock.get_fps()
        self._frame = timestamp.frame
        self._elapsed_seconds = timestamp.elapsed_seconds

    def update(self,
               sensor_display,
               world,
               timestamp,
               messages={},
               actions={},
               reasonings={},
        ):
        self._current_frame = sensor_display
        self.on_world_tick(timestamp)
        self._info_text = [
            f'Server: {self._server_fps:.2f} FPS',
            f'Client: {pygame.time.Clock().get_fps():.2f} FPS',
            f'Frame: {self._frame}',
            f'Elapsed: {self._elapsed_seconds:.2f} s',
        ]
        if messages:
            self._info_text += [
                'Messages',
            ]
            for vehicle_id, message in messages.items():
                self._info_text += [
                    f'Vehicle {vehicle_id} : {message.language_message}',
                ]
        if actions:
            self._info_text += [
                'Actions',
            ]
            for vehicle_id, action in actions.items():
                self._info_text += [
                    f'Vehicle {vehicle_id}',
                ]
                if action:
                    for key, value in action.items():
                        self._info_text += [
                            f'  {key}: {value}',
                        ]
        self.render()

    def render(self):
        camera_surface = pygame.surfarray.make_surface(self._current_frame.swapaxes(0, 1)[:,:,::-1])
        self.display.blit(camera_surface, (420,0))
        info_surface = pygame.Surface((420, self.dim[1]))
        info_surface.set_alpha(100)
        self.display.blit(info_surface, (0, 0))
        v_offset = 4
        bar_h_offset = 100
        bar_width = 106
        for item in self._info_text:
            if v_offset + 18 > self.dim[1]:
                break
            if isinstance(item, list):
                if len(item) > 1:
                    points = [(x + 8, v_offset + 8 + (1.0 - y) * 30) for x, y in enumerate(item)]
                    pygame.draw.lines(self.display, (255, 136, 0), False, points, 2)
                item = None
                v_offset += 18
            elif isinstance(item, tuple):
                if isinstance(item[1], bool):
                    rect = pygame.Rect((bar_h_offset, v_offset + 8), (6, 6))
                    pygame.draw.rect(self.display, (255, 255, 255), rect, 0 if item[1] else 1)
                else:
                    rect_border = pygame.Rect((bar_h_offset, v_offset + 8), (bar_width, 6))
                    pygame.draw.rect(self.display, (0, 255, 0), rect_border, 1)
                    f = (item[1] - item[2]) / (item[3] - item[2])
                    if item[2] < 0.0:
                        rect = pygame.Rect((bar_h_offset + f * (bar_width - 6), v_offset + 8), (6, 6))
                    else:
                        rect = pygame.Rect((bar_h_offset, v_offset + 8), (f * bar_width, 6))
                    pygame.draw.rect(self.display, (255, 255, 255), rect)
                item = item[0]
            if item:  # At this point has to be a str.
                surface, offset = self.text_wrap(item)
                self.display.blit(surface, (8, v_offset))
                v_offset += offset
            v_offset += 18
        pygame.image.save(self.display, self.logdir + f'hud_{self._frame}.png')
    
    def text_wrap(self, text, max_width=400):
        lines = []
        words = text.split(' ')
        text_margin = 8
        while len(words) > 0:
            i = 1
            while self.font.size(' '.join(words[:i]))[0] + text_margin < max_width and i < len(words):
                i += 1
            lines.append(' '.join(words[:i]))
            words = words[i:]
        surface = pygame.Surface((400, len(lines) * 18), pygame.SRCALPHA)
        for i, line in enumerate(lines):
            surface.blit(self.font.render(line, True, (255, 255, 255)), (0, i * 18))
        v_offset = i * 18
        return surface, v_offset 